//==============================================
//  Detector de quintas/octavas paralelas v0.1
//
//  Copyright (C)2015 Jörn Eichler (heuchi)
//  Traducción al español (C)2025
//
//  Este programa es software libre: puede redistribuirlo y/o modificarlo
//  bajo los términos de la Licencia Pública General GNU publicada por
//  la Free Software Foundation, ya sea la versión 3 de la Licencia, o
//  (a su elección) cualquier versión posterior.
//
//  Este programa se distribuye con la esperanza de que sea útil,
//  pero SIN NINGUNA GARANTÍA; sin siquiera la garantía implícita de
//  COMERCIABILIDAD o IDONEIDAD PARA UN PROPÓSITO PARTICULAR. Consulte la
//  Licencia Pública General GNU para más detalles.
//
//  Debería haber recibido una copia de la Licencia Pública General GNU
//  junto con este programa. Si no, vea <http://www.gnu.org/licenses/>.
//==============================================
/*ACTUALIZACIÓN POR Corentin Richard y Chat-GPT
 * SPDX-License-Identifier: GPL-3.0-only
 * MuseScore-Studio-CLA-applies
 *
 * MuseScore Studio
 * Composición y Notación Musical
 *
 * Copyright (C) 2021 MuseScore Limited
 *
 * Este programa es software libre: puede redistribuirlo y/o modificarlo
 * bajo los términos de la Licencia Pública General GNU publicada por
 * la Free Software Foundation.
 *
 * Este programa se distribuye con la esperanza de que sea útil,
 * pero SIN NINGUNA GARANTÍA; sin siquiera la garantía implícita de
 * COMERCIABILIDAD o IDONEIDAD PARA UN PROPÓSITO PARTICULAR. Consulte la
 * Licencia Pública General GNU para más detalles.
 *
 * Debería haber recibido una copia de la Licencia Pública General GNU
 * junto con este programa. Si no, vea <https://www.gnu.org/licenses/>.
 *
 ==============================================
 * VERSIÓN EN ESPAÑOL POR PAUL VELASCO 2025
 * Universidad De Las Artes del Ecuador.
 *
 * MuseScore Studio
 * Composición y Notación Musical
 *
 * Copyright (C) 2021 MuseScore Limited
 *
 * Este programa es software libre: puede redistribuirlo y/o modificarlo
 * bajo los términos de la Licencia Pública General GNU publicada por
 * la Free Software Foundation.
 *
 * Este programa se distribuye con la esperanza de que sea útil,
 * pero SIN NINGUNA GARANTÍA; sin siquiera la garantía implícita de
 * COMERCIABILIDAD o IDONEIDAD PARA UN PROPÓSITO PARTICULAR. Consulte la
 * Licencia Pública General GNU para más detalles.
 */

import QtQuick 2.15
import MuseScore 3.0

MuseScore {
      title: "Detector de Paralelismos Objetables"
      menuPath: "Plugins.Corrección.Detector de quintas/octavas paralelas"
      description: "Detecta quintas y octavas paralelas.\nMarca quintas y octavas consecutivas y también paralelos directos ascendentes."
      version: "0.2"
      requiresScore: true
      thumbnailName: "parallels-es.jpg"

      property var colorQuinta: "#ff6500";
      property var colorOctava: "#ff0050";
      property var colorDirecto: "#a03500";

      property bool procesarTodo: false;
      property bool acordesError: false;

      // Para contar cada tipo de paralelo encontrado
      property int quintasParalelas: 0;
      property int octavasParalelas: 0;
      property int quintasDirectas: 0;
      property int octavasDirectas: 0;

      id: detectarParalelos

      Component.onCompleted: {
          if (mscoreMajorVersion >= 4) {
              detectarParalelos.title = "Detector de Quintas/Octavas Paralelas";
              detectarParalelos.categoryCode = "herramientas-composicion-arreglos"
          }
      }

      function signo(x) {
            if (x > 0) return(1);
            else if (x == 0) return(0);
            else return(-1);
      }

      function marcarColor(nota1, nota2, color) {
            nota1.color = color;
            nota2.color = color;
      }

      function marcarTexto(nota1, nota2, msg, color, pista, tick) {
            marcarColor(nota1, nota2, color);
            var miTexto = newElement(Element.STAFF_TEXT);
            
            // Mejorar formato del texto: cada palabra con mayúscula inicial
            var palabras = msg.split(" ");
            for (var i = 0; i < palabras.length; i++) {
                  palabras[i] = palabras[i].charAt(0).toUpperCase() + palabras[i].slice(1);
            }
            miTexto.text = palabras.join(" ");
            
            miTexto.offsetY = 1;
            
            // Añadir formato adicional
            miTexto.bold = true;
            miTexto.color = color;
            
            var cursor = curScore.newCursor();
            cursor.rewind(0);
            cursor.track = pista;
            while (cursor.tick < tick) {
                  cursor.next();
            }
            cursor.add(miTexto);
      }

      onRun: {
            console.log("inicio")
            if (typeof curScore == 'undefined' || curScore == null) {
                  console.log("no se encontró partitura");
                  quit();
            }

            curScore.startCmd();
            
            // Reiniciar contadores
            quintasParalelas = 0;
            octavasParalelas = 0;
            quintasDirectas = 0;
            octavasDirectas = 0;

            // buscar selección
            var pentagramaInicio;
            var pentagramaFin;
            var tickFin;

            var cursor = curScore.newCursor();
            cursor.rewind(1);
            if (!cursor.segment) {
                  // sin selección
                  console.log("sin selección: procesando toda la partitura");
                  procesarTodo = true;
                  pentagramaInicio = 0;
                  pentagramaFin = curScore.nstaves;
            } else {
                  pentagramaInicio = cursor.staffIdx;
                  cursor.rewind(2);
                  pentagramaFin = cursor.staffIdx + 1;
                  tickFin = cursor.tick;
                  if (tickFin == 0) {
                        // la selección incluye el final de la partitura
                        tickFin = curScore.lastSegment.tick + 1;
                  }
                  cursor.rewind(1);
                  console.log("La selección es: Pentagramas("+pentagramaInicio+"-"+pentagramaFin+") Ticks("+cursor.tick+"-"+tickFin+")");
            }

            // inicializar estructura de datos
            var cambiado = [];
            var notaActual = [];
            var notaAnterior = [];
            var silencioActual = [];
            var silencioAnterior = [];
            var tickActual = [];
            var tickAnterior = [];

            var paralelosEncontrados = 0;

            var pista;
            var pistaInicio = pentagramaInicio * 4;
            var pistaFin = pentagramaFin * 4;

            for (pista = pistaInicio; pista < pistaFin; pista++) {
                  silencioActual[pista] = true;
                  silencioAnterior[pista] = true;
                  cambiado[pista] = false;
                  notaActual[pista] = 0;
                  notaAnterior[pista] = 0;
                  tickActual[pista] = 0;
                  tickAnterior[pista] = 0;
            }

            if (procesarTodo) {
                  cursor.track = 0;
                  cursor.rewind(0);
            } else {
                  cursor.rewind(1);
            }

            var segmento = cursor.segment;

            while (segmento && (procesarTodo || segmento.tick < tickFin)) {
                  // Paso 1: leer notas
                  for (pista = pistaInicio; pista < pistaFin; pista++) {
                        if (segmento.elementAt(pista)) {
                              if (segmento.elementAt(pista).type == Element.CHORD) {
                                    var notas = segmento.elementAt(pista).notes;

                                    if (notas.length > 1) {
                                          console.log("¡encontrado acorde con más de una nota!");
                                          acordesError = true;
                                    }

                                    var nota = notas[notas.length - 1];

                                    tickAnterior[pista] = tickActual[pista];
                                    silencioAnterior[pista] = silencioActual[pista];
                                    notaAnterior[pista] = notaActual[pista];
                                    silencioActual[pista] = false;
                                    notaActual[pista] = nota;
                                    tickActual[pista] = segmento.tick;
                                    cambiado[pista] = true;
                              } else if (segmento.elementAt(pista).type == Element.REST) {
                                    if (!silencioActual[pista]) {
                                          silencioAnterior[pista] = silencioActual[pista];
                                          notaAnterior[pista] = notaActual[pista];
                                          silencioActual[pista] = true;
                                          cambiado[pista] = false;
                                    }
                              } else {
                                    cambiado[pista] = false;
                              }
                        } else {
                              cambiado[pista] = false;
                        }
                  }

                  // Paso 2: encontrar paralelos
                  for (pista = pistaInicio; pista < pistaFin; pista++) {
                        var i;
                        if (cambiado[pista] && (!silencioAnterior[pista])) {
                              var dir1 = signo(notaActual[pista].pitch - notaAnterior[pista].pitch);
                              if (dir1 == 0) continue;
                              for (i = pista + 1; i < pistaFin; i++) {
                                    if (cambiado[i] && (!silencioAnterior[i])) {
                                          var dir2 = signo(notaActual[i].pitch - notaAnterior[i].pitch);
                                          if (dir1 == dir2) {
                                                var cint = notaActual[pista].pitch - notaActual[i].pitch;
                                                var pint = notaAnterior[pista].pitch - notaAnterior[i].pitch;
                                                if (Math.abs(cint % 12) == 7) {
                                                      if (cint == pint) {
                                                            paralelosEncontrados++;
                                                            quintasParalelas++;
                                                            marcarTexto(notaAnterior[pista], notaAnterior[i], "quinta paralela",
                                                                  colorQuinta, pista, tickAnterior[pista]);
                                                            marcarColor(notaActual[pista], notaActual[i], colorQuinta);
                                                      } else if (dir1 == 1 && Math.abs(pint) < Math.abs(cint)) {
                                                            paralelosEncontrados++;
                                                            quintasDirectas++;
                                                            marcarTexto(notaAnterior[pista], notaAnterior[i], "quinta directa",
                                                                  colorDirecto, pista, tickAnterior[pista]);
                                                            marcarColor(notaActual[pista], notaActual[i], colorDirecto);
                                                      }
                                                }
                                                if (Math.abs(cint % 12) == 0) {
                                                      if (cint == pint) {
                                                            paralelosEncontrados++;
                                                            octavasParalelas++;
                                                            marcarTexto(notaAnterior[pista], notaAnterior[i], "octava paralela",
                                                                  colorOctava, pista, tickAnterior[pista]);
                                                            marcarColor(notaActual[pista], notaActual[i], colorOctava);
                                                      } else if (dir1 == 1 && Math.abs(pint) < Math.abs(cint)) {
                                                            paralelosEncontrados++;
                                                            octavasDirectas++;
                                                            marcarTexto(notaAnterior[pista], notaAnterior[i], "octava directa",
                                                                  colorDirecto, pista, tickAnterior[pista]);
                                                            marcarColor(notaActual[pista], notaActual[i], colorDirecto);
                                                      }
                                                }
                                          }
                                    }
                              }
                        }
                  }
                  segmento = segmento.next;
            }

            curScore.endCmd();
            console.log("finalizado");
            
            // Crear mensaje pop-up con el resumen de los resultados encontrados
            var mensaje = "";
            if (paralelosEncontrados > 0) {
                  mensaje = "Análisis completado:\n\n";
                  if (quintasParalelas > 0) 
                        mensaje += "• " + quintasParalelas + " Quinta" + (quintasParalelas > 1 ? "s" : "") + " Paralela" + (quintasParalelas > 1 ? "s" : "") + "\n";
                  if (octavasParalelas > 0) 
                        mensaje += "• " + octavasParalelas + " Octava" + (octavasParalelas > 1 ? "s" : "") + " Paralela" + (octavasParalelas > 1 ? "s" : "") + "\n";
                  if (quintasDirectas > 0) 
                        mensaje += "• " + quintasDirectas + " Quinta" + (quintasDirectas > 1 ? "s" : "") + " Directa" + (quintasDirectas > 1 ? "s" : "") + "\n";
                  if (octavasDirectas > 0) 
                        mensaje += "• " + octavasDirectas + " Octava" + (octavasDirectas > 1 ? "s" : "") + " Directa" + (octavasDirectas > 1 ? "s" : "") + "\n";
                  
                  // Añadir totales
                  mensaje += "\nTotal: " + paralelosEncontrados + " Movimiento" + (paralelosEncontrados > 1 ? "s" : "") + " Problemático" + (paralelosEncontrados > 1 ? "s" : "") + " encontrado" + (paralelosEncontrados > 1 ? "s" : "") + ".";
            } else {
                  mensaje = "¡Felicidades! No se encontraron Movimientos Paralelos prohibidos en esta partitura.";
            }
            
            // Mostrar cuadro de diálogo con los resultados
            Qt.createQmlObject('import QtQuick.Dialogs 1.2; MessageDialog { 
                  title: "Detector de Paralelos - Resultados", 
                  text: "' + mensaje.replace(/\n/g, "\\n") + '", 
                  icon: StandardIcon.Information,
                  onAccepted: { Qt.quit() } 
            }', detectarParalelos, "messageDialog").open();
            
            quit();
      }
}